@extends('layouts.app')

@section('title', 'Gráficos de Estações')

@section('content')
<div class="space-y-6">
    <!-- Header -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between">
            <div>
                <h1 class="text-2xl font-bold text-gray-800 flex items-center">
                    <i class="fas fa-satellite-dish mr-3 text-green-600"></i>
                    Gráficos de Estações
                </h1>
                <p class="text-gray-600 mt-2">Monitoramento de precipitação, temperatura e dados hidrométricos</p>
            </div>
            <div class="text-right">
                <div class="text-3xl font-bold text-green-600">{{ $estacoes->count() }}</div>
                <div class="text-sm text-gray-500">Estações Monitoradas</div>
            </div>
        </div>
    </div>

    <!-- Filtros de Visualização -->
    <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
        <div class="bg-gradient-to-r from-green-50 to-green-100 border-b border-gray-200 px-6 py-4">
            <div class="flex items-center justify-between">
                <div class="flex items-center">
                    <i class="fas fa-filter text-green-600 mr-3"></i>
                    <h2 class="text-lg font-semibold text-gray-800">Filtros de Visualização</h2>
                </div>
                <button onclick="atualizarGraficos()"
                        class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-lg text-sm font-medium flex items-center transition-all duration-200 shadow-sm hover:shadow-md">
                    <i class="fas fa-sync mr-2"></i>
                    Atualizar Gráficos
                </button>
            </div>
        </div>

        <div class="p-6">
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                <div class="space-y-2">
                    <label class="block text-sm font-medium text-gray-700 flex items-center">
                        <i class="fas fa-calendar-alt text-green-500 mr-2"></i>
                        Período de Análise
                    </label>
                    <select id="periodo" class="w-full border border-gray-300 rounded-lg px-4 py-3 text-sm focus:border-green-500 focus:ring-2 focus:ring-green-200 bg-white shadow-sm transition-all duration-200">
                        <option value="7">Últimos 7 dias</option>
                        <option value="30" selected>Últimos 30 dias</option>
                        <option value="90">Últimos 3 meses</option>
                        <option value="365">Último ano</option>
                    </select>
                </div>

                <div class="space-y-2">
                    <label class="block text-sm font-medium text-gray-700 flex items-center">
                        <i class="fas fa-layer-group text-green-500 mr-2"></i>
                        Tipo de Estação
                    </label>
                    <select id="tipo_estacao" class="w-full border border-gray-300 rounded-lg px-4 py-3 text-sm focus:border-green-500 focus:ring-2 focus:ring-green-200 bg-white shadow-sm transition-all duration-200">
                        <option value="">Todos os tipos</option>
                        <option value="hidrometrica">Estação Hidrométrica</option>
                        <option value="pluviometrica">Estação Pluviométrica</option>
                        <option value="meteorologica">Estação Meteorológica</option>
                        <option value="climatologica">Estação Climatológica</option>
                    </select>
                </div>

                <div class="space-y-2">
                    <label class="block text-sm font-medium text-gray-700 flex items-center">
                        <i class="fas fa-satellite-dish text-green-500 mr-2"></i>
                        Estação Específica
                    </label>
                    <select id="estacao_filter" class="w-full border border-gray-300 rounded-lg px-4 py-3 text-sm focus:border-green-500 focus:ring-2 focus:ring-green-200 bg-white shadow-sm transition-all duration-200">
                        <option value="">Todas as Estações</option>
                        @foreach($estacoes as $estacao)
                            <option value="{{ $estacao->id }}" data-tipo="{{ $estacao->tipo }}">{{ $estacao->nome }}</option>
                        @endforeach
                    </select>
                </div>

                <div class="space-y-2">
                    <label class="block text-sm font-medium text-gray-700 flex items-center">
                        <i class="fas fa-chart-line text-green-500 mr-2"></i>
                        Métrica de Visualização
                    </label>
                    <select id="metrica" class="w-full border border-gray-300 rounded-lg px-4 py-3 text-sm focus:border-green-500 focus:ring-2 focus:ring-green-200 bg-white shadow-sm transition-all duration-200">
                        <option value="nivel_hidrometrico">Nível Hidrométrico (m)</option>
                        <option value="precipitacao_mm">Precipitação (mm)</option>
                        <option value="temperatura_max">Temperatura Máxima (°C)</option>
                        <option value="temperatura_min">Temperatura Mínima (°C)</option>
                        <option value="caudal">Caudal (m³/s)</option>
                    </select>
                </div>
            </div>

            <div class="mt-6 pt-4 border-t border-gray-100">
                <div class="flex items-center justify-between text-sm text-gray-600">
                    <div class="flex items-center">
                        <i class="fas fa-info-circle mr-2"></i>
                        <span>Os filtros são aplicados automaticamente. Selecione um tipo de estação para filtrar as métricas disponíveis</span>
                    </div>
                    <div class="flex items-center space-x-4">
                        <button onclick="limparFiltros()" class="text-gray-500 hover:text-gray-700 flex items-center">
                            <i class="fas fa-undo mr-1"></i>
                            Limpar Filtros
                        </button>
                        <button onclick="exportarDados()" class="text-green-600 hover:text-green-700 flex items-center">
                            <i class="fas fa-download mr-1"></i>
                            Exportar
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Resumo das Estações -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        @php
            $totalEstacoes = $estacoes->count();
            $estacoesComLeituras = $estacoes->filter(fn($e) => $e->ultima_leitura)->count();
            $hidrometricas = $estacoes->where('tipo', 'hidrometrica')->count();
            $pluviometricas = $estacoes->where('tipo', 'pluviometrica')->count();
        @endphp

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-satellite-dish text-green-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Total de Estações</p>
                    <p class="text-2xl font-bold text-gray-900">{{ $totalEstacoes }}</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-chart-line text-blue-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Com Leituras</p>
                    <p class="text-2xl font-bold text-gray-900">{{ $estacoesComLeituras }}</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-tint text-purple-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Hidrométricas</p>
                    <p class="text-2xl font-bold text-gray-900">{{ $hidrometricas }}</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-cyan-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-cloud-rain text-cyan-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Pluviométricas</p>
                    <p class="text-2xl font-bold text-gray-900">{{ $pluviometricas }}</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Gráfico Principal -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between mb-6">
            <h2 class="text-lg font-semibold text-gray-800">Evolução Temporal</h2>
            <div class="flex items-center space-x-2">
                <span class="text-sm text-gray-500">Última atualização:</span>
                <span class="text-sm font-medium text-gray-700" id="last-update">{{ now()->format('d/m/Y H:i') }}</span>
            </div>
        </div>
        <div class="h-96">
            <canvas id="grafico-principal"></canvas>
        </div>
    </div>

    <!-- Gráfico Auxiliar (para múltiplas métricas) -->
    <div class="bg-white rounded-lg shadow p-6" id="grafico-auxiliar-container" style="display: none;">
        <h2 class="text-lg font-semibold text-gray-800 mb-6">Dados Complementares</h2>
        <div class="h-64">
            <canvas id="grafico-auxiliar"></canvas>
        </div>
    </div>

    <!-- Lista de Estações -->
    <div class="bg-white rounded-lg shadow">
        <div class="px-6 py-4 border-b border-gray-200">
            <h2 class="text-lg font-semibold text-gray-800">Estado Atual das Estações</h2>
        </div>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Estação</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tipo</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Última Leitura</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nível/Precipitação</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Temperatura</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Estado</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Ações</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @foreach($estacoes as $estacao)
                    <tr class="hover:bg-gray-50">
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="flex items-center">
                                <div class="flex-shrink-0 h-8 w-8">
                                    <div class="h-8 w-8 rounded-full
                                        @if($estacao->tipo == 'hidrometrica') bg-purple-100
                                        @elseif($estacao->tipo == 'pluviometrica') bg-cyan-100
                                        @else bg-green-100 @endif
                                        flex items-center justify-center">
                                        <i class="
                                            @if($estacao->tipo == 'hidrometrica') fas fa-tint text-purple-600
                                            @elseif($estacao->tipo == 'pluviometrica') fas fa-cloud-rain text-cyan-600
                                            @else fas fa-thermometer-half text-green-600 @endif
                                            text-xs"></i>
                                    </div>
                                </div>
                                <div class="ml-3">
                                    <div class="text-sm font-medium text-gray-900">{{ $estacao->nome }}</div>
                                    <div class="text-sm text-gray-500">{{ $estacao->codigo }}</div>
                                </div>
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium
                                @if($estacao->tipo == 'hidrometrica') bg-purple-100 text-purple-800
                                @elseif($estacao->tipo == 'pluviometrica') bg-cyan-100 text-cyan-800
                                @elseif($estacao->tipo == 'meteorologica') bg-green-100 text-green-800
                                @else bg-gray-100 text-gray-800 @endif">
                                {{ ucfirst($estacao->tipo) }}
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            @if($estacao->ultima_leitura)
                                {{ $estacao->ultima_leitura->data_leitura->format('d/m/Y') }}
                            @else
                                <span class="text-gray-400">Sem dados</span>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            @if($estacao->ultima_leitura)
                                @if($estacao->tipo == 'hidrometrica' && $estacao->ultima_leitura->nivel_hidrometrico)
                                    {{ number_format($estacao->ultima_leitura->nivel_hidrometrico, 2) }} m
                                @elseif(in_array($estacao->tipo, ['pluviometrica', 'meteorologica']) && $estacao->ultima_leitura->precipitacao_mm)
                                    {{ number_format($estacao->ultima_leitura->precipitacao_mm, 1) }} mm
                                @else
                                    <span class="text-gray-400">-</span>
                                @endif
                            @else
                                <span class="text-gray-400">-</span>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            @if($estacao->ultima_leitura && ($estacao->ultima_leitura->temperatura_max || $estacao->ultima_leitura->temperatura_min))
                                <div class="text-xs">
                                    @if($estacao->ultima_leitura->temperatura_max)
                                        <div class="text-red-600">Máx: {{ number_format($estacao->ultima_leitura->temperatura_max, 1) }}°C</div>
                                    @endif
                                    @if($estacao->ultima_leitura->temperatura_min)
                                        <div class="text-blue-600">Mín: {{ number_format($estacao->ultima_leitura->temperatura_min, 1) }}°C</div>
                                    @endif
                                </div>
                            @else
                                <span class="text-gray-400">-</span>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium
                                @if($estacao->estado == 'activa') bg-green-100 text-green-800
                                @elseif($estacao->estado == 'inactiva') bg-red-100 text-red-800
                                @else bg-yellow-100 text-yellow-800 @endif">
                                {{ ucfirst($estacao->estado) }}
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium space-x-2">
                            <a href="{{ route('charts.estacao-detail', $estacao) }}"
                               class="text-green-600 hover:text-green-900">
                                <i class="fas fa-chart-line mr-1"></i>Gráfico
                            </a>
                            <a href="{{ route('estacoes.show', $estacao) }}"
                               class="text-gray-600 hover:text-gray-900">
                                <i class="fas fa-eye mr-1"></i>Detalhes
                            </a>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Scripts -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
let grafico = null;
let graficoAuxiliar = null;
const cores = [
    '#10B981', '#3B82F6', '#F59E0B', '#EF4444', '#8B5CF6',
    '#06B6D4', '#F97316', '#84CC16', '#EC4899', '#6366F1'
];

// Inicializar gráfico
document.addEventListener('DOMContentLoaded', function() {
    atualizarGraficos();

    // Filtrar estações por tipo
    document.getElementById('tipo_estacao').addEventListener('change', function() {
        const tipoSelecionado = this.value;
        const selectEstacao = document.getElementById('estacao_filter');
        const options = selectEstacao.querySelectorAll('option');

        options.forEach(option => {
            if (option.value === '') {
                option.style.display = 'block';
                return;
            }

            const tipoEstacao = option.getAttribute('data-tipo');
            if (!tipoSelecionado || tipoEstacao === tipoSelecionado) {
                option.style.display = 'block';
            } else {
                option.style.display = 'none';
            }
        });

        selectEstacao.value = '';
        atualizarMetricasDisponiveis(tipoSelecionado);
    });
});

function atualizarMetricasDisponiveis(tipo) {
    const metricaSelect = document.getElementById('metrica');
    metricaSelect.innerHTML = '';

    if (!tipo || tipo === 'hidrometrica') {
        metricaSelect.innerHTML += '<option value="nivel_hidrometrico">Nível Hidrométrico</option>';
        metricaSelect.innerHTML += '<option value="caudal">Caudal (m³/s)</option>';
    }

    if (!tipo || ['pluviometrica', 'meteorologica', 'climatologica'].includes(tipo)) {
        metricaSelect.innerHTML += '<option value="precipitacao_mm">Precipitação (mm)</option>';
    }

    if (!tipo || ['meteorologica', 'climatologica'].includes(tipo)) {
        metricaSelect.innerHTML += '<option value="temperatura_max">Temperatura Máxima</option>';
        metricaSelect.innerHTML += '<option value="temperatura_min">Temperatura Mínima</option>';
    }
}

async function atualizarGraficos() {
    const periodo = document.getElementById('periodo').value;
    const estacaoId = document.getElementById('estacao_filter').value;
    const metrica = document.getElementById('metrica').value;
    const tipoEstacao = document.getElementById('tipo_estacao').value;

    try {
        if (estacaoId) {
            // Gráfico de uma estação específica
            const response = await fetch(`/graficos/api/estacao/${estacaoId}/data?days=${periodo}`);
            const data = await response.json();
            criarGraficoUnicoEstacao(data, metrica, estacaoId);
        } else {
            // Gráfico comparativo por tipo ou todas
            criarGraficoComparativo(metrica, periodo, tipoEstacao);
        }

        document.getElementById('last-update').textContent = new Date().toLocaleString('pt-PT', {
            day: '2-digit',
            month: '2-digit',
            year: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    } catch (error) {
        console.error('Erro ao carregar dados:', error);
    }
}

function criarGraficoUnicoEstacao(data, metrica, estacaoId) {
    const ctx = document.getElementById('grafico-principal').getContext('2d');

    if (grafico) {
        grafico.destroy();
    }

    const labels = data.map(d => new Date(d.data_leitura).toLocaleDateString('pt-PT'));
    const valores = data.map(d => parseFloat(d[metrica]) || 0);

    // Para estações hidrométricas, mostrar também os níveis por horário
    let datasets = [{
        label: getMetricaLabel(metrica),
        data: valores,
        borderColor: cores[0],
        backgroundColor: cores[0] + '20',
        fill: true,
        tension: 0.4
    }];

    // Se for nível hidrométrico, adicionar os horários
    if (metrica === 'nivel_hidrometrico' && data.length > 0) {
        const temNiveis = data.some(d => d.nivel_6h || d.nivel_9h || d.nivel_12h || d.nivel_15h || d.nivel_18h);

        if (temNiveis) {
            const horarios = ['6h', '9h', '12h', '15h', '18h'];
            horarios.forEach((hora, index) => {
                const valoresHora = data.map(d => parseFloat(d[`nivel_${hora}`]) || null);
                if (valoresHora.some(v => v !== null)) {
                    datasets.push({
                        label: `Nível ${hora}`,
                        data: valoresHora,
                        borderColor: cores[index + 1],
                        backgroundColor: 'transparent',
                        fill: false,
                        tension: 0.4,
                        pointRadius: 3
                    });
                }
            });
        }
    }

    grafico = new Chart(ctx, {
        type: 'line',
        data: {
            labels: labels,
            datasets: datasets
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                title: {
                    display: true,
                    text: `${getMetricaLabel(metrica)} - Estação Selecionada`
                }
            },
            scales: {
                y: {
                    beginAtZero: metrica === 'precipitacao_mm',
                    ticks: {
                        callback: function(value) {
                            return getMetricaUnit(metrica, value);
                        }
                    }
                }
            }
        }
    });
}

function criarGraficoComparativo(metrica, periodo, tipoEstacao) {
    const ctx = document.getElementById('grafico-principal').getContext('2d');

    if (grafico) {
        grafico.destroy();
    }

    // Filtrar estações
    let estacoesFiltradas = @json($estacoes->values());

    if (tipoEstacao) {
        estacoesFiltradas = estacoesFiltradas.filter(e => e.tipo === tipoEstacao);
    }

    const labels = estacoesFiltradas.map(e => e.nome);
    const valores = estacoesFiltradas.map(e => {
        if (e.ultima_leitura) {
            return parseFloat(e.ultima_leitura[metrica]) || 0;
        }
        return 0;
    });

    grafico = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{
                label: getMetricaLabel(metrica),
                data: valores,
                backgroundColor: cores.slice(0, estacoesFiltradas.length),
                borderColor: cores.slice(0, estacoesFiltradas.length),
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                title: {
                    display: true,
                    text: `Comparação: ${getMetricaLabel(metrica)} ${tipoEstacao ? `(${tipoEstacao})` : ''}`
                }
            },
            scales: {
                y: {
                    beginAtZero: metrica === 'precipitacao_mm',
                    ticks: {
                        callback: function(value) {
                            return getMetricaUnit(metrica, value);
                        }
                    }
                }
            }
        }
    });
}

function getMetricaLabel(metrica) {
    const labels = {
        'nivel_hidrometrico': 'Nível Hidrométrico',
        'precipitacao_mm': 'Precipitação',
        'temperatura_max': 'Temperatura Máxima',
        'temperatura_min': 'Temperatura Mínima',
        'caudal': 'Caudal'
    };
    return labels[metrica] || metrica;
}

function getMetricaUnit(metrica, value) {
    const units = {
        'nivel_hidrometrico': value + ' m',
        'precipitacao_mm': value + ' mm',
        'temperatura_max': value + '°C',
        'temperatura_min': value + '°C',
        'caudal': value + ' m³/s'
    };
    return units[metrica] || value;
}

function limparFiltros() {
    document.getElementById('periodo').value = '30';
    document.getElementById('tipo_estacao').value = '';
    document.getElementById('estacao_filter').value = '';
    document.getElementById('metrica').value = 'nivel_hidrometrico';

    // Resetar as opções de estações
    const selectEstacao = document.getElementById('estacao_filter');
    const options = selectEstacao.querySelectorAll('option');
    options.forEach(option => {
        option.style.display = 'block';
    });

    // Resetar métricas disponíveis
    atualizarMetricasDisponiveis('');
    atualizarGraficos();
}

function exportarDados() {
    const periodo = document.getElementById('periodo').value;
    const estacaoId = document.getElementById('estacao_filter').value;
    const metrica = document.getElementById('metrica').value;
    const tipoEstacao = document.getElementById('tipo_estacao').value;

    // Construir URL de exportação
    let url = '/graficos/api/estacoes/export?';
    url += `periodo=${periodo}&metrica=${metrica}`;

    if (estacaoId) {
        url += `&estacao_id=${estacaoId}`;
    }

    if (tipoEstacao) {
        url += `&tipo=${tipoEstacao}`;
    }

    // Abrir em nova aba para download
    window.open(url, '_blank');
}
</script>
@endsection