@extends('layouts.app')

@section('title', 'Estações')

@section('content')
<div class="space-y-6">
    <!-- Header -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between">
            <div>
                <h1 class="text-2xl font-bold text-gray-800">
                    <i class="fas fa-satellite-dish mr-3 text-green-600"></i>
                    Estações de Monitoramento
                </h1>
                <p class="text-gray-600 mt-2">Gerir todas as estações do sistema hidrológico</p>
            </div>
            <div class="flex space-x-4">
                <a href="{{ route('estacoes.create') }}" 
                   class="bg-green-600 text-white px-4 py-2 rounded-md hover:bg-green-700 transition-colors">
                    <i class="fas fa-plus mr-2"></i>
                    Nova Estação
                </a>
                <a href="#" 
                   class="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700 transition-colors">
                    <i class="fas fa-upload mr-2"></i>
                    Importar Dados
                </a>
            </div>
        </div>
    </div>

    <!-- Filters -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-6">
            <form method="GET" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Buscar</label>
                    <input type="text" name="search" value="{{ request('search') }}" 
                           placeholder="Nome ou código da estação"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Tipo</label>
                    <select name="tipo" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500">
                        <option value="">Todos</option>
                        <option value="hidrometrica" {{ request('tipo') == 'hidrometrica' ? 'selected' : '' }}>Hidrométrica</option>
                        <option value="pluviometrica" {{ request('tipo') == 'pluviometrica' ? 'selected' : '' }}>Pluviométrica</option>
                        <option value="meteorologica" {{ request('tipo') == 'meteorologica' ? 'selected' : '' }}>Meteorológica</option>
                        <option value="climatologica" {{ request('tipo') == 'climatologica' ? 'selected' : '' }}>Climatológica</option>
                    </select>
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Estado</label>
                    <select name="estado" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500">
                        <option value="">Todos</option>
                        <option value="activa" {{ request('estado') == 'activa' ? 'selected' : '' }}>Ativa</option>
                        <option value="inactiva" {{ request('estado') == 'inactiva' ? 'selected' : '' }}>Inativa</option>
                        <option value="manutencao" {{ request('estado') == 'manutencao' ? 'selected' : '' }}>Manutenção</option>
                    </select>
                </div>
                <div class="flex items-end">
                    <button type="submit" class="w-full bg-gray-600 text-white px-4 py-2 rounded-md hover:bg-gray-700 transition-colors">
                        <i class="fas fa-search mr-2"></i>
                        Filtrar
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Estações Table -->
    <div class="bg-white rounded-lg shadow overflow-hidden">
        <div class="px-6 py-4 border-b border-gray-200">
            <h3 class="text-lg font-medium text-gray-900">
                Lista de Estações ({{ $estacoes->total() }})
            </h3>
        </div>
        
        @if($estacoes->count() > 0)
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Estação & Características</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tipo & Equipamento</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Localização & Coordenadas</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Última Leitura & Dados</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Estado & Status</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Ações</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        @foreach($estacoes as $estacao)
                        <tr class="hover:bg-gray-50">
                            <td class="px-6 py-4">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 bg-green-100 rounded-full flex items-center justify-center">
                                        @if($estacao->tipo == 'hidrometrica')
                                            <i class="fas fa-tint text-green-600"></i>
                                        @elseif($estacao->tipo == 'pluviometrica') 
                                            <i class="fas fa-cloud-rain text-green-600"></i>
                                        @elseif($estacao->tipo == 'meteorologica')
                                            <i class="fas fa-cloud-sun text-green-600"></i>
                                        @else
                                            <i class="fas fa-satellite-dish text-green-600"></i>
                                        @endif
                                    </div>
                                    <div class="ml-4 min-w-0 flex-1">
                                        <div class="text-sm font-medium text-gray-900">
                                            <a href="{{ route('estacoes.show', $estacao) }}" 
                                               class="hover:text-green-600">
                                                {{ $estacao->nome }}
                                            </a>
                                        </div>
                                        <div class="text-sm text-gray-500">{{ $estacao->codigo }}</div>
                                        @if($estacao->rio)
                                            <div class="text-xs text-green-600 mt-1">
                                                <i class="fas fa-water mr-1"></i>Rio {{ $estacao->rio }}
                                            </div>
                                        @endif
                                        @if($estacao->altitude)
                                            <div class="text-xs text-blue-600 mt-1">
                                                <i class="fas fa-mountain mr-1"></i>{{ number_format($estacao->altitude, 0) }}m altitude
                                            </div>
                                        @endif
                                        @if($estacao->data_instalacao)
                                            <div class="text-xs text-gray-500 mt-1">
                                                <i class="fas fa-calendar mr-1"></i>Instalada em {{ $estacao->data_instalacao->format('Y') }}
                                            </div>
                                        @endif
                                    </div>
                                </div>
                            </td>
                            
                            <td class="px-6 py-4">
                                @php
                                    $tipoConfig = [
                                        'hidrometrica' => ['class' => 'bg-blue-100 text-blue-800', 'icon' => 'fas fa-tint', 'text' => 'Hidrométrica'],
                                        'pluviometrica' => ['class' => 'bg-green-100 text-green-800', 'icon' => 'fas fa-cloud-rain', 'text' => 'Pluviométrica'],
                                        'meteorologica' => ['class' => 'bg-purple-100 text-purple-800', 'icon' => 'fas fa-cloud-sun', 'text' => 'Meteorológica'],
                                        'climatologica' => ['class' => 'bg-orange-100 text-orange-800', 'icon' => 'fas fa-thermometer-half', 'text' => 'Climatológica']
                                    ];
                                    $config = $tipoConfig[$estacao->tipo] ?? ['class' => 'bg-gray-100 text-gray-800', 'icon' => 'fas fa-satellite-dish', 'text' => ucfirst($estacao->tipo)];
                                @endphp
                                <div class="space-y-2">
                                    <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium {{ $config['class'] }}">
                                        <i class="{{ $config['icon'] }} mr-2"></i>
                                        {{ $config['text'] }}
                                    </span>
                                    @if($estacao->equipamento)
                                        <div class="text-xs text-gray-700 bg-gray-50 px-2 py-1 rounded">
                                            <i class="fas fa-microchip mr-1"></i>{{ $estacao->equipamento }}
                                        </div>
                                    @endif
                                    @if($estacao->sistema_automatizado)
                                        <div class="text-xs text-green-700 bg-green-50 px-2 py-1 rounded">
                                            <i class="fas fa-robot mr-1"></i>Sistema Automatizado
                                        </div>
                                    @endif
                                    @if($estacao->telemetria)
                                        <div class="text-xs text-blue-700 bg-blue-50 px-2 py-1 rounded">
                                            <i class="fas fa-satellite mr-1"></i>Telemetria Ativa
                                        </div>
                                    @endif
                                    @if($estacao->frequencia_monitoramento)
                                        <div class="text-xs text-purple-600">
                                            <i class="fas fa-clock mr-1"></i>{{ ucfirst($estacao->frequencia_monitoramento) }}
                                        </div>
                                    @endif
                                </div>
                            </td>

                            <td class="px-6 py-4">
                                <div class="space-y-1">
                                    <div class="text-sm text-gray-900">
                                        <i class="fas fa-map-marker-alt text-gray-400 mr-1"></i>
                                        {{ $estacao->provincia ?? 'N/A' }}
                                    </div>
                                    <div class="text-sm text-gray-500">{{ $estacao->distrito ?? 'N/A' }}</div>
                                    @if($estacao->posto_administrativo)
                                        <div class="text-xs text-gray-600">
                                            <i class="fas fa-building mr-1"></i>{{ $estacao->posto_administrativo }}
                                        </div>
                                    @endif
                                    @if($estacao->latitude && $estacao->longitude)
                                        <div class="text-xs text-blue-600 bg-blue-50 px-2 py-1 rounded">
                                            <i class="fas fa-globe mr-1"></i>
                                            {{ number_format($estacao->latitude, 4) }}°, {{ number_format($estacao->longitude, 4) }}°
                                        </div>
                                    @endif
                                    @if($estacao->bacia_hidrografica)
                                        <div class="text-xs text-green-600">
                                            <i class="fas fa-water mr-1"></i>{{ $estacao->bacia_hidrografica->nome }}
                                        </div>
                                    @endif
                                </div>
                            </td>
                            
                            <td class="px-6 py-4">
                                @if($estacao->ultima_leitura)
                                    <div class="space-y-2">
                                        <div class="text-sm text-gray-900">
                                            <i class="fas fa-clock text-green-500 mr-1"></i>
                                            {{ $estacao->ultima_leitura->data_leitura->format('d/m/Y') }}
                                            @if($estacao->ultima_leitura->hora_leitura)
                                                <span class="text-xs text-gray-500">{{ $estacao->ultima_leitura->hora_leitura }}</span>
                                            @endif
                                        </div>
                                        
                                        <!-- Dados Meteorológicos -->
                                        @if($estacao->ultima_leitura->precipitacao_mm)
                                            <div class="text-xs text-white bg-blue-500 px-2 py-1 rounded">
                                                <i class="fas fa-cloud-rain mr-1"></i>
                                                {{ number_format($estacao->ultima_leitura->precipitacao_mm, 1) }}mm chuva
                                            </div>
                                        @endif
                                        
                                        @if($estacao->ultima_leitura->temperatura_max || $estacao->ultima_leitura->temperatura_min)
                                            <div class="flex space-x-1">
                                                @if($estacao->ultima_leitura->temperatura_max)
                                                    <div class="text-xs text-red-700 bg-red-100 px-2 py-1 rounded">
                                                        <i class="fas fa-thermometer-half mr-1"></i>
                                                        {{ number_format($estacao->ultima_leitura->temperatura_max, 1) }}°C máx
                                                    </div>
                                                @endif
                                                @if($estacao->ultima_leitura->temperatura_min)
                                                    <div class="text-xs text-blue-700 bg-blue-100 px-2 py-1 rounded">
                                                        {{ number_format($estacao->ultima_leitura->temperatura_min, 1) }}°C mín
                                                    </div>
                                                @endif
                                            </div>
                                        @endif
                                        
                                        <!-- Dados Hidrológicos -->
                                        @if($estacao->ultima_leitura->nivel_6h || $estacao->ultima_leitura->nivel_12h || $estacao->ultima_leitura->nivel_18h)
                                            <div class="text-xs text-cyan-700 bg-cyan-100 px-2 py-1 rounded">
                                                <i class="fas fa-water mr-1"></i>
                                                @if($estacao->ultima_leitura->nivel_6h)
                                                    6h: {{ number_format($estacao->ultima_leitura->nivel_6h, 2) }}m
                                                @endif
                                                @if($estacao->ultima_leitura->nivel_12h)
                                                    12h: {{ number_format($estacao->ultima_leitura->nivel_12h, 2) }}m
                                                @endif
                                                @if($estacao->ultima_leitura->nivel_18h)
                                                    18h: {{ number_format($estacao->ultima_leitura->nivel_18h, 2) }}m
                                                @endif
                                            </div>
                                        @endif
                                        
                                        @if($estacao->ultima_leitura->humidade_relativa)
                                            <div class="text-xs text-green-700 bg-green-100 px-2 py-1 rounded">
                                                <i class="fas fa-tint mr-1"></i>
                                                {{ number_format($estacao->ultima_leitura->humidade_relativa, 0) }}% humidade
                                            </div>
                                        @endif
                                        
                                        <div class="text-xs text-gray-500">
                                            <i class="fas fa-{{ $estacao->ultima_leitura->metodo_leitura == 'automatico' ? 'robot' : 'user' }} mr-1"></i>
                                            {{ ucfirst($estacao->ultima_leitura->metodo_leitura ?? 'Manual') }}
                                            @if($estacao->ultima_leitura->validado)
                                                <i class="fas fa-check-circle text-green-500 ml-1" title="Validado"></i>
                                            @else
                                                <i class="fas fa-clock text-orange-500 ml-1" title="Pendente"></i>
                                            @endif
                                        </div>
                                    </div>
                                @else
                                    <div class="text-center py-4">
                                        <div class="text-sm text-gray-500">
                                            <i class="fas fa-minus-circle mr-1"></i>
                                            Sem dados
                                        </div>
                                        <div class="text-xs text-gray-400 mt-1">Nenhuma leitura registrada</div>
                                    </div>
                                @endif
                            </td>
                            
                            <td class="px-6 py-4">
                                @php
                                    $estadoConfig = [
                                        'activa' => ['class' => 'bg-green-100 text-green-800', 'icon' => 'fas fa-circle-check', 'text' => 'Ativa'],
                                        'inactiva' => ['class' => 'bg-red-100 text-red-800', 'icon' => 'fas fa-circle-xmark', 'text' => 'Inativa'],
                                        'manutencao' => ['class' => 'bg-yellow-100 text-yellow-800', 'icon' => 'fas fa-screwdriver-wrench', 'text' => 'Manutenção']
                                    ];
                                    $estadoConf = $estadoConfig[$estacao->estado] ?? ['class' => 'bg-gray-100 text-gray-800', 'icon' => 'fas fa-question-circle', 'text' => ucfirst($estacao->estado)];
                                @endphp
                                <div class="space-y-2">
                                    <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium {{ $estadoConf['class'] }}">
                                        <i class="{{ $estadoConf['icon'] }} mr-2"></i>
                                        {{ $estadoConf['text'] }}
                                    </span>
                                    
                                    @if($estacao->nivel_alerta && $estacao->ultima_leitura)
                                        @php
                                            $alerta = false;
                                            if(($estacao->ultima_leitura->nivel_6h && $estacao->ultima_leitura->nivel_6h >= $estacao->nivel_alerta) ||
                                               ($estacao->ultima_leitura->nivel_12h && $estacao->ultima_leitura->nivel_12h >= $estacao->nivel_alerta) ||
                                               ($estacao->ultima_leitura->nivel_18h && $estacao->ultima_leitura->nivel_18h >= $estacao->nivel_alerta)) {
                                                $alerta = true;
                                            }
                                        @endphp
                                        @if($alerta)
                                            <div class="text-xs text-red-700 bg-red-100 px-2 py-1 rounded border border-red-300 animate-pulse">
                                                <i class="fas fa-exclamation-triangle mr-1"></i>
                                                NÍVEL DE ALERTA
                                            </div>
                                        @endif
                                    @endif
                                    
                                    @if($estacao->ultima_leitura)
                                        @php
                                            $ultimaAtualizacao = $estacao->ultima_leitura->data_leitura;
                                            $diasSemAtualizacao = $ultimaAtualizacao->diffInDays(now());
                                            $statusClass = 'bg-green-100 text-green-800';
                                            $statusIcon = 'fas fa-check-circle';
                                            $statusText = 'Atualizado';
                                            
                                            if ($diasSemAtualizacao > 7) {
                                                $statusClass = 'bg-red-100 text-red-800';
                                                $statusIcon = 'fas fa-exclamation-circle';
                                                $statusText = 'Desatualizado';
                                            } elseif ($diasSemAtualizacao > 3) {
                                                $statusClass = 'bg-yellow-100 text-yellow-800';
                                                $statusIcon = 'fas fa-clock';
                                                $statusText = 'Atrasado';
                                            }
                                        @endphp
                                        <div class="text-xs {{ $statusClass }} px-2 py-1 rounded">
                                            <i class="{{ $statusIcon }} mr-1"></i>
                                            {{ $statusText }}
                                            @if($diasSemAtualizacao > 0)
                                                <div class="mt-1">({{ $diasSemAtualizacao }} dias)</div>
                                            @endif
                                        </div>
                                    @else
                                        <div class="text-xs bg-gray-100 text-gray-800 px-2 py-1 rounded">
                                            <i class="fas fa-minus-circle mr-1"></i>
                                            Sem dados
                                        </div>
                                    @endif
                                </div>
                            </td>
                            
                            <td class="px-6 py-4 text-sm font-medium">
                                <div class="space-y-2">
                                    <!-- Ações Principais -->
                                    <div class="flex items-center space-x-1">
                                        <a href="{{ route('estacoes.show', $estacao) }}"
                                           class="inline-flex items-center px-2 py-1 border border-transparent text-xs leading-4 font-medium rounded text-white bg-blue-600 hover:bg-blue-700 transition-colors"
                                           title="Visualizar detalhes">
                                            <i class="fas fa-eye mr-1"></i>
                                            Ver
                                        </a>
                                        <a href="{{ route('estacoes.edit', $estacao) }}"
                                           class="inline-flex items-center px-2 py-1 border border-transparent text-xs leading-4 font-medium rounded text-white bg-green-600 hover:bg-green-700 transition-colors"
                                           title="Editar">
                                            <i class="fas fa-edit mr-1"></i>
                                            Editar
                                        </a>
                                    </div>

                                    <!-- Ações de Dados -->
                                    <div class="flex items-center space-x-1">
                                        <a href="{{ route('estacoes.leituras', $estacao) }}"
                                           class="inline-flex items-center px-2 py-1 border border-gray-300 text-xs leading-4 font-medium rounded text-gray-700 bg-white hover:bg-gray-50 transition-colors"
                                           title="Ver leituras">
                                            <i class="fas fa-chart-line mr-1"></i>
                                            Dados
                                        </a>
                                        <a href="{{ route('estacoes.leituras.create', $estacao) }}"
                                           class="inline-flex items-center px-2 py-1 border border-purple-300 text-xs leading-4 font-medium rounded text-purple-700 bg-purple-50 hover:bg-purple-100 transition-colors"
                                           title="Adicionar leitura">
                                            <i class="fas fa-plus mr-1"></i>
                                            Nova Leitura
                                        </a>
                                    </div>

                                    <!-- Dropdown para mais ações -->
                                    <div class="relative">
                                        <button onclick="toggleDropdown('actions-{{ $estacao->id }}')" 
                                                class="inline-flex items-center px-2 py-1 border border-gray-300 text-xs leading-4 font-medium rounded text-gray-700 bg-white hover:bg-gray-50 transition-colors w-full justify-center">
                                            <i class="fas fa-ellipsis-h mr-1"></i>
                                            Mais
                                        </button>
                                        <div id="actions-{{ $estacao->id }}" class="hidden absolute right-0 mt-1 w-48 bg-white rounded-md shadow-lg z-10 border">
                                            <div class="py-1">
                                                <a href="{{ route('estacoes.exportar-leituras', $estacao) }}" class="block px-4 py-2 text-xs text-gray-700 hover:bg-gray-100">
                                                    <i class="fas fa-download mr-2"></i>
                                                    Exportar dados
                                                </a>
                                                <a href="{{ route('charts.estacao-detail', $estacao) }}" class="block px-4 py-2 text-xs text-gray-700 hover:bg-gray-100">
                                                    <i class="fas fa-chart-bar mr-2"></i>
                                                    Estatísticas
                                                </a>
                                                <a href="{{ route('estacoes.leituras.import', $estacao) }}" class="block px-4 py-2 text-xs text-gray-700 hover:bg-gray-100">
                                                    <i class="fas fa-upload mr-2"></i>
                                                    Importar dados
                                                </a>
                                                <div class="border-t border-gray-100">
                                                    <form method="POST" action="{{ route('estacoes.destroy', $estacao) }}" 
                                                          class="inline w-full" onsubmit="return confirm('Tem certeza que deseja excluir esta estação?')">
                                                        @csrf
                                                        @method('DELETE')
                                                        <button type="submit" class="block w-full text-left px-4 py-2 text-xs text-red-700 hover:bg-red-50">
                                                            <i class="fas fa-trash mr-2"></i>
                                                            Excluir estação
                                                        </button>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </td>
                        </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            @if($estacoes->hasPages())
            <div class="px-6 py-4 border-t border-gray-200">
                {{ $estacoes->withQueryString()->links() }}
            </div>
            @endif
        @else
            <div class="text-center py-12">
                <div class="w-24 h-24 bg-gradient-to-br from-green-100 to-green-200 rounded-full flex items-center justify-center mx-auto mb-4 shadow-lg">
                    <i class="fas fa-satellite-dish text-4xl text-green-600"></i>
                </div>
                <h3 class="text-lg font-medium text-gray-900 mb-2">Nenhuma estação encontrada</h3>
                <p class="text-gray-500 mb-6 max-w-md mx-auto">
                    @if(request()->filled('search') || request()->filled('tipo') || request()->filled('estado'))
                        Nenhuma estação corresponde aos filtros aplicados. Tente ajustar os critérios de busca.
                    @else
                        Comece adicionando sua primeira estação ao sistema para começar o monitoramento hidrológico.
                    @endif
                </p>
                <div class="flex flex-col sm:flex-row gap-3 justify-center">
                    @if(request()->filled('search') || request()->filled('tipo') || request()->filled('estado'))
                        <a href="{{ route('estacoes.index') }}" 
                           class="inline-flex items-center px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 transition-colors">
                            <i class="fas fa-times mr-2"></i>
                            Limpar Filtros
                        </a>
                    @endif
                    <a href="{{ route('estacoes.create') }}" 
                       class="inline-flex items-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-green-600 hover:bg-green-700 transition-colors">
                        <i class="fas fa-plus mr-2"></i>
                        Adicionar Estação
                    </a>
                </div>
            </div>
        @endif
    </div>
</div>

<script>
function toggleDropdown(dropdownId) {
    const dropdown = document.getElementById(dropdownId);
    const isHidden = dropdown.classList.contains('hidden');
    
    // Fechar todos os outros dropdowns
    document.querySelectorAll('[id^="actions-"]').forEach(el => {
        if (el.id !== dropdownId) {
            el.classList.add('hidden');
        }
    });
    
    // Toggle do dropdown atual
    if (isHidden) {
        dropdown.classList.remove('hidden');
    } else {
        dropdown.classList.add('hidden');
    }
}

// Fechar dropdown ao clicar fora
document.addEventListener('click', function(event) {
    if (!event.target.closest('[onclick*="toggleDropdown"]') && !event.target.closest('[id^="actions-"]')) {
        document.querySelectorAll('[id^="actions-"]').forEach(el => {
            el.classList.add('hidden');
        });
    }
});
</script>
@endsection